// app/detail/[id]/page.jsx
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { getListingById } from "@/service/ListingService";
import { Anchor, Cog, Compass, DoorOpen, Hash, Layers, Layout, Package, Palette, Ruler, Scale, Search, Settings2, Wrench, Zap } from "lucide-react";
import { Separator } from "@/components/ui/separator";
import Link from "next/link";
import {
    Calendar,
    Gauge,
    Settings,
    Fuel,
    MapPin,
    Phone,
    Mail,
    User
} from "lucide-react";
import ListingMediaViewer from "@/components/listing/listingMediaViewer";

export default async function ListingDetailPage({ params }) {
    const { id } = await params;

    let listing;

    try {
        listing = await getListingById(id);
    } catch (error) {
        //error UI
        return (
            <div className="min-h-screen bg-gray-50 py-8">
                <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                    <Card>
                        <CardContent className="p-12 text-center">
                            <h1 className="text-3xl font-bold mb-4">Truck Not Found</h1>
                            <p className="text-xl text-gray-600">The truck you're looking for doesn't exist or has been removed.</p>
                        </CardContent>
                        <Link href="/find-trucks" className="text-center">
                            <Button>
                                <Search className="mr-2 h-4 w-4" />
                                View All Trucks
                            </Button>
                        </Link>
                    </Card>
                </div>
            </div>
        );
    }

    const formatMileage = (mileage) => {
        return new Intl.NumberFormat('en-US').format(mileage);
    };

    const formatPrice = (price) => {
        return new Intl.NumberFormat('en-US', {
            style: 'currency',
            currency: 'USD',
            minimumFractionDigits: 0,
            maximumFractionDigits: 0,
        }).format(price);
    };

    return (
        <div className="min-h-screen bg-gray-50 py-8">
            <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">

                <div className="grid grid-cols-1 lg:grid-cols-3 gap-8">
                    {/* Left Column - Images and Details */}
                    <div className="lg:col-span-2 space-y-6">
                        {/* Main Image */}
                        <div className="relative w-full h-96 mb-6">
                            <ListingMediaViewer
                                images={listing.images || []}
                                videos={listing.videos || []}
                                title={listing.title}
                            />

                        </div>
                        {/* Vehicle Details */}
                        <Card>
                            <CardHeader>
                                <CardTitle className="text-primary font-semibold text-2xl">{listing.title}</CardTitle>
                            </CardHeader>
                            <CardContent>
                                <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                                    {/* Common Fields */}
                                    <div className="flex items-center">
                                        <Calendar className="text-gray-600 mr-3 h-5 w-5" />
                                        <div>
                                            <div className="font-medium">Year</div>
                                            <div className="text-gray-600">{listing.year || "N/A"}</div>
                                        </div>
                                    </div>

                                    <div className="flex items-center">
                                        <MapPin className="text-gray-600 mr-3 h-5 w-5" />
                                        <div>
                                            <div className="font-medium">Location</div>
                                            <div className="text-gray-600">
                                                {[listing.address, listing.city, listing.state, listing.country]
                                                    .filter(Boolean)
                                                    .join(", ") || "N/A"}
                                            </div>
                                        </div>
                                    </div>

                                    {/* TRUCK FIELDS */}
                                    {listing.type === "TRUCK" && (
                                        <>
                                            <div className="flex items-center">
                                                <Gauge className="text-gray-600 mr-3 h-5 w-5" />
                                                <div>
                                                    <div className="font-medium">Mileage</div>
                                                    <div className="text-gray-600">{formatMileage(listing.truckMileage) || "N/A"}</div>
                                                </div>
                                            </div>

                                            <div className="flex items-center">
                                                <Hash className="text-gray-600 mr-3 h-5 w-5" />
                                                <div>
                                                    <div className="font-medium">VIN</div>
                                                    <div className="text-gray-600">{listing.truckVinNumber || "N/A"}</div>
                                                </div>
                                            </div>

                                            <div className="flex items-center">
                                                <Zap className="text-gray-600 mr-3 h-5 w-5" />
                                                <div>
                                                    <div className="font-medium">Horse Power</div>
                                                    <div className="text-gray-600">{listing.truckHorsePower || "N/A"}</div>
                                                </div>
                                            </div>

                                            <div className="flex items-center">
                                                <Package className="text-gray-600 mr-3 h-5 w-5" />
                                                <div>
                                                    <div className="font-medium">Stock Number</div>
                                                    <div className="text-gray-600">{listing.truckStockNumber || "N/A"}</div>
                                                </div>
                                            </div>

                                            <div className="flex items-center">
                                                <Cog className="text-gray-600 mr-3 h-5 w-5" />
                                                <div>
                                                    <div className="font-medium">Engine Manufacturer</div>
                                                    <div className="text-gray-600">{listing.truckEngineManufacturer || "N/A"}</div>
                                                </div>
                                            </div>

                                            <div className="flex items-center">
                                                <Wrench className="text-gray-600 mr-3 h-5 w-5" />
                                                <div>
                                                    <div className="font-medium">Engine</div>
                                                    <div className="text-gray-600">{listing.engine || "N/A"}</div>
                                                </div>
                                            </div>

                                            <div className="flex items-center">
                                                <Settings2 className="text-gray-600 mr-3 h-5 w-5" />
                                                <div>
                                                    <div className="font-medium">Transmission - Speed</div>
                                                    <div className="text-gray-600">{`${listing.transmission} - ${listing.transmissionSpeed}` || "N/A"}</div>
                                                </div>
                                            </div>

                                            <div className="flex items-center">
                                                <Compass className="text-gray-600 mr-3 h-5 w-5" />
                                                <div>
                                                    <div className="font-medium">Driver Side</div>
                                                    <div className="text-gray-600">{listing.truckDriverSide || "N/A"}</div>
                                                </div>
                                            </div>

                                            <div className="flex items-center">
                                                <Settings className="text-gray-600 mr-3 h-5 w-5" />
                                                <div>
                                                    <div className="font-medium">Class</div>
                                                    <div className="text-gray-600">{listing.truckClassName || "N/A"}</div>
                                                </div>
                                            </div>

                                            <div className="flex items-center">
                                                <Scale className="text-gray-600 mr-3 h-5 w-5" />
                                                <div>
                                                    <div className="font-medium">GVWR</div>
                                                    <div className="text-gray-600">{listing.truckClassGvwr || "N/A"}</div>
                                                </div>
                                            </div>

                                            <div className="flex items-center">
                                                <Fuel className="text-gray-600 mr-3 h-5 w-5" />
                                                <div>
                                                    <div className="font-medium">Fuel Type</div>
                                                    <div className="text-gray-600">{listing.truckFuelType || "N/A"}</div>
                                                </div>
                                            </div>
                                        </>
                                    )}

                                    {/* TRAILER FIELDS */}
                                    {listing.type === "TRAILER" && (
                                        <>
                                            <div className="flex items-center">
                                                <Settings className="text-gray-600 mr-3 h-5 w-5" />
                                                <div>
                                                    <div className="font-medium">Trailer Type</div>
                                                    <div className="text-gray-600">{listing.trailerType || "N/A"}</div>
                                                </div>
                                            </div>

                                            <div className="flex items-center">
                                                <Hash className="text-gray-600 mr-3 h-5 w-5" />
                                                <div>
                                                    <div className="font-medium">VIN</div>
                                                    <div className="text-gray-600">{listing.trailerVinNumber || "N/A"}</div>
                                                </div>
                                            </div>

                                            <div className="flex items-center">
                                                <Package className="text-gray-600 mr-3 h-5 w-5" />
                                                <div>
                                                    <div className="font-medium">Stock Number</div>
                                                    <div className="text-gray-600">{listing.trailerStockNumber || "N/A"}</div>
                                                </div>
                                            </div>

                                            <div className="flex items-center">
                                                <Ruler className="text-gray-600 mr-3 h-5 w-5" />
                                                <div>
                                                    <div className="font-medium">Length</div>
                                                    <div className="text-gray-600">{listing.trailerLength || "N/A"}</div>
                                                </div>
                                            </div>

                                            <div className="flex items-center">
                                                <Ruler className="text-gray-600 mr-3 h-5 w-5" />
                                                <div>
                                                    <div className="font-medium">Width</div>
                                                    <div className="text-gray-600">{listing.trailerWidth || "N/A"}</div>
                                                </div>
                                            </div>

                                            <div className="flex items-center">
                                                <Palette className="text-gray-600 mr-3 h-5 w-5" />
                                                <div>
                                                    <div className="font-medium">Color</div>
                                                    <div className="text-gray-600">{listing.trailerColor || "N/A"}</div>
                                                </div>
                                            </div>

                                            <div className="flex items-center">
                                                <Ruler className="text-gray-600 mr-3 h-5 w-5" />
                                                <div>
                                                    <div className="font-medium">Height</div>
                                                    <div className="text-gray-600">{listing.trailerHeight || "N/A"}</div>
                                                </div>
                                            </div>

                                            <div className="flex items-center">
                                                <Layers className="text-gray-600 mr-3 h-5 w-5" />
                                                <div>
                                                    <div className="font-medium">Composition</div>
                                                    <div className="text-gray-600">{listing.trailerComposition || "N/A"}</div>
                                                </div>
                                            </div>

                                            <div className="flex items-center">
                                                <Gauge className="text-gray-600 mr-3 h-5 w-5" />
                                                <div>
                                                    <div className="font-medium">Number of Axles</div>
                                                    <div className="text-gray-600">{listing.trailerNumberOfAxles || "N/A"}</div>
                                                </div>
                                            </div>

                                            <div className="flex items-center">
                                                <Settings className="text-gray-600 mr-3 h-5 w-5" />
                                                <div>
                                                    <div className="font-medium">Axle Configuration</div>
                                                    <div className="text-gray-600">{listing.trailerAxleConfiguration || "N/A"}</div>
                                                </div>
                                            </div>

                                            <div className="flex items-center">
                                                <Anchor className="text-gray-600 mr-3 h-5 w-5" />
                                                <div>
                                                    <div className="font-medium">Hitch</div>
                                                    <div className="text-gray-600">{listing.trailerHitch || "N/A"}</div>
                                                </div>
                                            </div>

                                            <div className="flex items-center">
                                                <Layout className="text-gray-600 mr-3 h-5 w-5" />
                                                <div>
                                                    <div className="font-medium">Floor Type</div>
                                                    <div className="text-gray-600">{listing.trailerFloorType || "N/A"}</div>
                                                </div>
                                            </div>

                                            <div className="flex items-center">
                                                <DoorOpen className="text-gray-600 mr-3 h-5 w-5" />
                                                <div>
                                                    <div className="font-medium">Number of Side Doors</div>
                                                    <div className="text-gray-600">{listing.trailerNumberOfSideDoors || "N/A"}</div>
                                                </div>
                                            </div>

                                            <div className="flex items-center">
                                                <DoorOpen className="text-gray-600 mr-3 h-5 w-5" />
                                                <div>
                                                    <div className="font-medium">Rear Opening</div>
                                                    <div className="text-gray-600">{listing.trailerRearOpening || "N/A"}</div>
                                                </div>
                                            </div>

                                            <div className="flex items-center">
                                                <Scale className="text-gray-600 mr-3 h-5 w-5" />
                                                <div>
                                                    <div className="font-medium">Trailer Weight (lbs)</div>
                                                    <div className="text-gray-600">{listing.trailerWeightLbs || "N/A"}</div>
                                                </div>
                                            </div>

                                            <div className="flex items-center">
                                                <Scale className="text-gray-600 mr-3 h-5 w-5" />
                                                <div>
                                                    <div className="font-medium">Capacity (lbs)</div>
                                                    <div className="text-gray-600">{listing.trailerCapacityLbs || "N/A"}</div>
                                                </div>
                                            </div>
                                        </>
                                    )}
                                </div>
                            </CardContent>

                        </Card>

                        {/* Description */}
                        <Card className="gap-4">
                            <CardHeader>
                                <CardTitle className="text-2xl">Description</CardTitle>
                            </CardHeader>
                            <CardContent>
                                <p className="text-gray-600 leading-relaxed whitespace-pre-line">
                                    {listing.description}
                                </p>
                            </CardContent>
                        </Card>
                    </div>

                    {/* Right Column - Pricing and Contact */}
                    <div className="space-y-6">
                        {/* Price and CTA */}
                        <Card>
                            <CardContent >
                                <div className="text-center mb-6">
                                    <div className="text-3xl font-bold text-primary mb-2">
                                        {formatPrice(listing.price)}
                                    </div>
                                </div>

                                <div className="space-y-3">
                                        {listing.plan.planName !== "Basic" && (
                                            <Button
                                                asChild
                                                size="lg"
                                                className="w-full bg-chart-5 hover:bg-chart-5/90"
                                            >
                                                <a
                                                    href={`tel:${[listing.seller.countryCode, listing.seller.phone]
                                                        .filter(Boolean)
                                                        .join("")}`}
                                                >
                                                    <Phone className="mr-2 h-4 w-4" />
                                                    Call Seller
                                                </a>
                                            </Button>
                                        )}

                                        <Button
                                            asChild
                                            size="lg"
                                            variant="outline"
                                            className="w-full"
                                        >
                                            <a href={`mailto:${listing.seller.email}`}>
                                                <Mail className="mr-2 h-4 w-4" />
                                                Email Seller
                                            </a>
                                        </Button>
                                </div>
                            </CardContent>
                        </Card>

                        {/* Seller Information */}
                        <Card className="gap-4">
                            <CardHeader>
                                <CardTitle className="flex items-center text-2xl">
                                    <User className="mr-2 h-5 w-5" />
                                    Seller Information
                                </CardTitle>
                            </CardHeader>
                            <CardContent>
                                <div className="space-y-4">
                                    <div>
                                        <div className="text-primary font-medium">{listing.seller.name}</div>
                                    </div>
                                    <Separator />
                                    <div className="space-y-2">
                                        {listing.plan.planName !== "Basic" && (
                                        <div className="flex items-center text-sm">
                                            <Phone className="text-gray-600 mr-2 h-4 w-4" />
                                            <span>{[listing.seller.countryCode, listing.seller.phone].filter(Boolean).join("-") || "N/A"}</span>
                                        </div>
                                        )}
                                        <div className="flex items-center text-sm">
                                            <Mail className="text-gray-600 mr-2 h-4 w-4" />
                                            <span>{listing.seller.email}</span>
                                        </div>
                                        <div className="flex items-center text-sm">
                                            <MapPin className="text-gray-600 mr-2 h-4 w-4" />
                                            <span>
                                                {[listing.seller.address, listing.seller.city, listing.seller.state, listing.seller.country]
                                                    .filter(Boolean)
                                                    .join(", ") || "N/A"}
                                            </span>
                                        </div>
                                    </div>
                                </div>
                            </CardContent>
                        </Card>

                        {/* Quick Facts */}
                        <Card>
                            <CardHeader>
                                <CardTitle className="text-2xl">Quick Facts</CardTitle>
                            </CardHeader>
                            <CardContent>
                                <div className="space-y-3 text-sm">
                                    <div className="flex justify-between">
                                        <span className="text-gray-600">Category</span>
                                        <span className="font-medium">{listing.categoryName}</span>
                                    </div>
                                    <div className="flex justify-between">
                                        <span className="text-gray-600">Maker</span>
                                        <span className="font-medium">{listing.makerName}</span>
                                    </div>
                                    <div className="flex justify-between">
                                        <span className="text-gray-600">Model</span>
                                        <span className="font-medium">{listing.modelName}</span>
                                    </div>
                                    <div className="flex justify-between">
                                        <span className="text-gray-600">Trim</span>
                                        <span className="font-medium">{listing.trimName}</span>
                                    </div>
                                    <div className="flex justify-between">
                                        <span className="text-gray-600">Year</span>
                                        <span className="font-medium">{listing.year}</span>
                                    </div>
                                </div>
                            </CardContent>
                        </Card>
                    </div>
                </div>
            </div>
        </div>
    );
}
